function setBvParams(bv)
%SETBVPARAMS Set parameters for a greenhouse crop model based on Vanthoor (2011)
% Inputs:
%   bv   - a StateSpaceModel object to be used as a Vanthoor model
%
% Based on the electronic appendices (the case of a Dutch greenhouse) of:
%   [1] Vanthoor, B., Stanghellini, C., van Henten, E. J. & de Visser, P. H. B. 
%       A methodology for model-based greenhouse design: Part 1, a greenhouse climate 
%       model for a broad range of designs and climates. Biosyst. Eng. 110, 363377 (2011).
%   [2] Vanthoor, B., de Visser, P. H. B., Stanghellini, C. & van Henten, E. J. 
%       A methodology for model-based greenhouse design: Part 2, description and 
%       validation of a tomato yield model. Biosyst. Eng. 110, 378-395 (2011).
% These are also available as Chapters 8 and 9, respecitvely, of
%   [3] Vanthoor, B. A model based greenhouse design method. (Wageningen University, 2011).
% Control decisions are based on:
% 	[4] Vanthoor, B., van Henten, E.J., Stanghellini, C., and de Visser, P.H.B. (2011). 
% 		A methodology for model-based greenhouse design: Part 3, sensitivity 
% 		analysis of a combined greenhouse climate-crop yield model. 
% 		Biosyst. Eng. 110, 396-412.
% 	[5] Vermeulen, P.C.M. (2016). Kwantitatieve informatie voor de glastuinbouw 
% 		2016-2017 (Bleiswijk), page V56-V58 Vine tomato
%   [6] Dueck, T., Elings, A., Kempkes, F., Knies, P., Garcia, N., Heij, G., 
% 		Janse, J., Kaarsemaker, R., Korsten, P., Maaswinkel, R., et al. (2004). 
% 		Energie in kengetallen : op zoek naar een nieuwe balans.
% Lamp parameters are based on:
%   [7] De Zwart, H.F., Baeza, E., Van Breugel, B., Mohammadkhani, V., 
%       and Janssen, H. (2017). De uitstralingmonitor. 52.
% Several other parameters are based on:
%   [8] Dueck, T., De Gelder, A., Janse, J., Kempkes, F., Baar, P.H., and 
%       Valstar, W. (2014). Het nieuwe belichten onder diffuus glas (Wageningen).
%   [9] Dueck, T. A., Janse, J., Eveleens, B. A., Kempkes, F. L. K., & Marcelis, L. F. M. (2012). 
%       Growth of Tomatoes under Hybrid LED and HPS Lighting. Acta Horticulturae, 1(952), 335342. 
%   [10]Nelson, J. A., & Bugbee, B. (2014). Economic Analysis of Greenhouse Lighting: 
%       Light Emitting Diodes vs. High Intensity Discharge Fixtures. PLoS ONE, 9(6), e99010. 
%   [11]Dueck, T., Janse, J., Schapendonk, A. H. C. M., Kempkes, F., 
%       Eveleens, B., Scheffers, K.,  Marcelis, L. F. M. (2010). 
%       Lichtbenuttig van tomaat onder LED en SON-T belichting. Wageningen.

% David Katzin, Wageningen University
% david.katzin@wur.nl


	%% Table 1 [1]
	% General model parameters

	%% Pg. 39 [1]

	% parameter 					            description 																					unit 					nominal value (source)
    addParam(bv, 'alfaLeafAir',5); 				% Convective heat exchange coefficient from the canopy leaf to the greenhouse air 				W m^{-2} K^{-1} 		5 [1]
	addParam(bv, 'L', 2.45e6); 					% Latent heat of evaporation 																	J kg^{-1}{water} 		2.45e6 [1]			
    addParam(bv, 'sigma', 5.67e-8); 			% Stefan-Boltzmann constant 																	W m^{-2} K^{-4} 		5.67e-8 [1]
	addParam(bv, 'epsCan', 1); 					% FIR emission coefficient of canopy 															-   					1 [1]
	addParam(bv, 'epsSky', 1); 					% FIR emission coefficient of the sky 															- 						1 [1]
	addParam(bv, 'etaGlobNir', 0.5); 			% Ratio of NIR in global radiation 																-	 					0.5 [1]
	addParam(bv, 'etaGlobPar', 0.5); 			% Ratio of PAR in global radiation 																-	 					0.5 [1]
	
	%% pg. 40 [1]
	% parameter 					            description 																					unit 					nominal value (source)
	addParam(bv, 'etaMgPpm', 0.554); 			% CO2 conversion factor from mg m^{-3} to ppm 													ppm mg^{-1} m^{3} 		0.554 [1]
	addParam(bv, 'etaRoofThr', 0.9); 			% Ratio between roof vent area and total vent area where no chimney effects is assumed 			- 						0.9 [1]
	addParam(bv, 'rhoAir0', 1.2); 				% Density of air at sea level 																	kg m^{-3} 				1.2 [1]
	addParam(bv, 'rhoCanPar', 0.07); 			% PAR reflection coefficient 																	- 						0.07 [1]
	addParam(bv, 'rhoCanNir', 0.35);  			% NIR reflection coefficient of the top of the canopy 											- 						0.35 [1]
	addParam(bv, 'rhoSteel', 7850); 			% Density of steel 																				kg m^{-3} 				7850 [1]
	addParam(bv, 'rhoWater', 1e3); 				% Density of water 																				kg m^{-3} 				1e3 [1]
	addParam(bv, 'gamma', 65.8); 				% Psychrometric constant 																		Pa K^{-1} 				65.8 [1]
	addParam(bv, 'omega', 1.99e-7); 			% Yearly frequency to calculate soil temperature 												s^{-1} 					1.99e7 [1]
	addParam(bv, 'capLeaf', 1.2e3); 			% Heat capacity of canopy leaves 																J K^{-1} m^{-2}{leaf} 	1.2e3 [1]
	addParam(bv, 'cEvap1', 4.3); 				% Coefficient for radiation effect on stomatal resistance 										W m^{-2} 				4.3 [1]
	addParam(bv, 'cEvap2', 0.54); 				% Coefficient for radiation effect on stomatal resistance 										W m^{-2} 				0.54 [1]
	
	%% pg. 41 [1]
	% parameter 					            description 																					unit 					nominal value (source)
    addParam(bv, 'cEvap3Day', 6.1e-7); 			% Coefficient for co2 effect on stomatal resistance (day) 										ppm^{-2} 				6.1e-7 [1]
	addParam(bv, 'cEvap3Night', 1.1e-11);		% Coefficient for co2 effect on stomatal resistance (night) 									ppm^{-2} 				1.1e-11 [1]
	addParam(bv, 'cEvap4Day', 4.3e-6); 			% Coefficient for vapor pressure effect on stomatal resistance (day) 							Pa^{-2} 				4.3e-6 [1]
	addParam(bv, 'cEvap4Night', 5.2e-6);		% Coefficient for vapor pressure effect on stomatal resistance (night) 							Pa^{-2} 				5.2e-6 [1]
	addParam(bv, 'cPAir', 1e3); 				% Specific heat capacity of air 																J K^{-1} kg^{-1} 		1e3 [1]
	addParam(bv, 'cPSteel', 0.64e3); 			% Specific heat capacity of steel 																J K^{-1} kg^{-1} 		0.64e3 [1]
	addParam(bv, 'cPWater', 4.18e3); 			% Specific heat capacity of water 																J K^{-1} kg^{-1} 		4.18e3 [1]
	addParam(bv, 'g', 9.81);                    % Acceleration of gravity 																		m s^{-2} 				9.81 [1]
	addParam(bv, 'hSo1', 0.04); 				% Thickness of soil layer 1 																	m 						0.04 [1]
	addParam(bv, 'hSo2', 0.08); 				% Thickness of soil layer 2 																	m 						0.08 [1]
	addParam(bv, 'hSo3', 0.16); 				% Thickness of soil layer 3 																	m 						0.16 [1]
	addParam(bv, 'hSo4', 0.32); 				% Thickness of soil layer 4 																	m 						0.32 [1]
	addParam(bv, 'hSo5', 0.64); 				% Thickness of soil layer 5 																	m 						0.64 [1]
	addParam(bv, 'k1Par', 0.7); 				% PAR extinction coefficient of the canopy 														- 						0.7 [1]
	addParam(bv, 'k2Par', 0.7); 				% PAR extinction coefficient of the canopy for light reflected from the floor 					- 						0.7 [1]
	addParam(bv, 'kNir', 0.27); 				% NIR extinction coefficient of the canopy 														- 						0.27 [1]
	addParam(bv, 'kFir', 0.94); 				% FIR extinction coefficient of the canopy 														- 						0.94 [1]
	addParam(bv, 'mAir', 28.96); 				% Molar mass of air 																			kg kmol^{-1} 			28.96 [1]
	
    addParam(bv, 'hSoOut', 1.28);               % Thickness of the external soil layer                                                          m                       1.28 (assumed)
    
	%% pg. 42 [1]
	% parameter 					            description 																					unit 					nominal value (source)
    addParam(bv, 'mWater', 18); 				% Molar mass of water 																			kg kmol^{-1} 			18 [1]
	addParam(bv, 'R', 8314); 					% Molar gas constant 																			J kmol^{-1} K^{-1} 		8314 [1]
	addParam(bv, 'rCanSp', 5); 					% Radiation value above the canopy when night becomes day 										W m^{-2} 				5 [1]
	addParam(bv, 'rB', 275); 					% Boundary layer resistance of the canopy for transpiration 									s m^{-1} 				275 [1]
	addParam(bv, 'rSMin', 82); 					% Minimum canopy resistance for transpiration 													s m^{-1} 				82 [1]
	addParam(bv, 'sRs', -1); 					% Slope of smoothed stomatal resistance model 													m W^{-2} 				-1 [1]
	addParam(bv, 'sMV12', -0.1); 				% Slope of smoothed condensation model 															Pa^{-1} 				-0.1 [1]
	
	%% Table 2 [1]
	% Location specific parameters - The Netherlands
	
	%% Construction properties [1]
	% parameter 					            description 																					unit 					nominal value (source)
    addParam(bv, 'etaGlobAir', 0.1); 			% Ratio of global radiation absorbed by the greenhouse construction 							- 						0.1 [1]
    addParam(bv, 'psi', 25);
    addParam(bv, 'aFlr', 1.4e4);                % Floor area of greenhouse 																		m^{2} 					1.4e4 [1]          
    addParam(bv, 'aCov', 1.8e4);                % Surface of the cover including side walls 													m^{2} 					1.8e4 [1]
    addParam(bv, 'hAir', 3.8);                  % Height of the main compartment 																m 						3.8 [1]
    addParam(bv, 'hGh', 4.2);                   % Mean height of the greenhouse 																m 						4.2 [1] 
    addParam(bv, 'cHecIn', 1.86); 				% Convective heat exchange between cover and outdoor air 										W m^{-2} K^{-1} 		1.86 [1]
	addParam(bv, 'cHecOut1', 2.8); 				% Convective heat exchange parameter between cover and outdoor air 								W m^{-2}{cover} K^{-1} 	2.8 [1]
	addParam(bv, 'cHecOut2', 1.2); 				% Convective heat exchange parameter between cover and outdoor air 								J m^{-3} K^{-1} 		1.2 [1]
	addParam(bv, 'cHecOut3', 1);	 			% Convective heat exchange parameter between cover and outdoor air 								- 					 	1 [1]
    addParam(bv, 'hElevation', 0);				% Altitude of greenhouse 																		m above sea level		0 [1]
	
	%% Ventilation properties [1]
	% parameter 					            description 																					unit 					nominal value (source)
    addParam(bv, 'aRoof', 1.4e3);               % Maximum roof ventilation area 																- 						0.1*aFlr [1]
    addParam(bv, 'hVent', 0.68);                % Vertical dimension of single ventilation opening 												m 						0.68 [1]
    addParam(bv, 'etaInsScr', 1); 				% Porosity of the insect screen 																- 						1 [1]
    addParam(bv, 'aSide', 0); 					% Side ventilation area 																		- 						0 [1]
	addParam(bv, 'cDgh', 0.75); 				% Ventilation discharge coefficient 															- 						0.75 [1]
    addParam(bv, 'cLeakage', 1e-4);
	addParam(bv, 'cWgh', 0.09); 				% Ventilation global wind pressure coefficient 													- 						0.09 [1]
	addParam(bv, 'hSideRoof', 0);               % Vertical distance between mid points of side wall and roof ventilation opening                m                       0 (no side ventilation)
    
	%% Roof [1]
	% parameter 					            description 																					unit 					nominal value (source)
	addParam(bv, 'epsRfFir', 0.85); 			% FIR emission coefficient of the roof 															- 						0.85 [1]
	addParam(bv, 'rhoRf', 2.6e3); 				% Density of the roof layer 																	kg m^{-3} 				2.6e3 [1]
	addParam(bv, 'rhoRfNir', 0.13); 			% NIR reflection coefficient of the roof 														- 						0.13 [1]
	addParam(bv, 'rhoRfPar', 0.13);  			% PAR reflection coefficient of the roof 														- 						0.13 [1]
	addParam(bv, 'rhoRfFir', 0.15);  			% FIR reflection coefficient of the roof 														- 						0.15 [1]
    addParam(bv, 'tauRfNir', 0.85); 			% NIR transmission coefficient of the roof 														- 						0.85 [1]
    addParam(bv, 'tauRfPar', 0.85);  			% PAR transmission coefficient of the roof 														- 						0.85 [1]
	addParam(bv, 'tauRfFir', 0);                % FIR transmission coefficient of the roof 														- 						0 [1]
	addParam(bv, 'lambdaRf', 1.05); 			% Thermal heat conductivity of the roof 														W m^{-1} K^{-1} 		1.05 [1]
	addParam(bv, 'cPRf', 0.84e3); 				% Specific heat capacity of roof layer 															J K^{-1} kg^{-1} 		0.84e3 [1]
    addParam(bv, 'hRf', 4e-3); 					% Thickness of roof layer 																		m 						4e-3 [1]
	
    
    %% Whitewash
    addParam(bv, 'epsShScrPerFir', 0);       	% FIR emission coefficient of the whitewash                                                     -                       0 (no whitewash)
    addParam(bv, 'rhoShScrPer', 0);             % Density of the whitewash                                                                      -                       0 (no whitewash)
    addParam(bv, 'rhoShScrPerNir', 0);       	% NIR reflection coefficient of whitewash                                                       -                       0 (no whitewash)
    addParam(bv, 'rhoShScrPerPar', 0);       	% PAR reflection coefficient of whitewash                                                       -                       0 (no whitewash)
    addParam(bv, 'rhoShScrPerFir', 0);       	% FIR reflection coefficient of whitewash                                                       -                       0 (no whitewash)
    addParam(bv, 'tauShScrPerNir', 1);       	% NIR transmission coefficient of whitewash                                                     -                       1 (no whitewash)
    addParam(bv, 'tauShScrPerPar', 1);       	% PAR transmission coefficient of whitewash                                                     -                       1 (no whitewash)
    addParam(bv, 'tauShScrPerFir', 1);       	% FIR transmission coefficient of whitewash                                                     -                       1 (no whitewash)
    addParam(bv, 'lambdaShScrPer', Inf);     	% Thermal heat conductivity of the whitewash                                                    W m^{-1} K^{-1}         Inf (no whitewash)
    addParam(bv, 'cPShScrPer', 0);              % Specific heat capacity of the whitewash                                                       J K^{-1} kg^{-1}        0 (no whitewash)
    addParam(bv, 'hShScrPer', 0);               % Thickness of the whitewash                                                                    m                       0 (no whitewash)
    
    %% Shadow screen 
    addParam(bv, 'rhoShScrNir', 0);     		% NIR reflection coefficient of shadow screen                                                   -                       0 (no shadow screen)
    addParam(bv, 'rhoShScrPar', 0);             % PAR reflection coefficient of shadow screen                                                   -                       0 (no shadow screen)
    addParam(bv, 'rhoShScrFir', 0);     		% FIR reflection coefficient of shadow screen                                                   -                       0 (no shadow screen)
    addParam(bv, 'tauShScrNir', 1);     		% NIR transmission coefficient of shadow screen                                                 -                       1 (no shadow screen)
    addParam(bv, 'tauShScrPar', 1);             % PAR transmission coefficient of shadow screen                                                 -                       1 (no shadow screen)
    addParam(bv, 'tauShScrFir', 1);     		% FIR transmission coefficient of shadow screen                                                 -                       1 (no shadow screen)
	addParam(bv, 'etaShScrCd', 0);      		% Effect of shadow screen on discharge coefficient                                              -                       0 (no shadow screen)
    addParam(bv, 'etaShScrCw', 0);      		% Effect of shadow screen on wind pressure coefficient                                          -                       0 (no shadow screen)
    addParam(bv, 'kShScr', 0);                  % Shadow screen flux coefficient 																m^{3} m^{-2} K^{-2/3} s^{-1} 0 (no shadow screen)

    
    %% Thermal screen [1]
	% parameter 					            description 																					unit 					nominal value (source)
	addParam(bv, 'epsThScrFir', 0.67); 			% FIR emissions coefficient of the thermal screen 												- 						0.67 [1]
	addParam(bv, 'rhoThScr', 0.2e3); 			% Density of thermal screen 																	kg m^{-3} 				0.2e3 [1]
	addParam(bv, 'rhoThScrNir', 0.35); 			% NIR reflection coefficient of thermal screen 													- 						0.35 [1]
	addParam(bv, 'rhoThScrPar', 0.35); 			% PAR reflection coefficient of thermal screen 													- 						0.35 [1]
	addParam(bv, 'rhoThScrFir', 0.18); 			% FIR reflection coefficient of thermal screen 													- 						0.18 [1]
	addParam(bv, 'tauThScrNir', 0.6); 			% NIR transmission coefficient of thermal screen 												- 						0.6 [1]
    addParam(bv, 'tauThScrPar', 0.6); 			% PAR transmission coefficient of thermal screen 												- 						0.6 [1]
    addParam(bv, 'tauThScrFir', 0.15); 			% FIR transmission coefficient of thermal screen 												- 						0.15 [1]
	addParam(bv, 'cPThScr', 1.8e3); 			% Specific heat capacity of thermal screen 														J kg^{-1} K^{-1} 		1.8e3 [1]
    addParam(bv, 'hThScr', 0.35e-3); 			% Thickness of thermal screen 																	m 						0.35e-3 [1]
    addParam(bv, 'kThScr', 0.05e-3); 			% Thermal screen flux coefficient 																m^{3} m^{-2} K^{-2/3} s^{-1} 0.05e-3 [1]
	
    %% Blackout screen [1]
	% parameter 					            description 																					unit 					nominal value (source)
	addParam(bv, 'epsBlScrFir', 0.67); 			% FIR emissions coefficient of the blackout screen 												- 						
	addParam(bv, 'rhoBlScr', 0.2e3); 			% Density of blackout screen 																	kg m^{-3} 				
	addParam(bv, 'rhoBlScrNir', 0.35); 			% NIR reflection coefficient of blackout screen 													- 						
	addParam(bv, 'rhoBlScrPar', 0.35); 			% PAR reflection coefficient of blackout screen 													- 						
	addParam(bv, 'rhoBlScrFir', 0.18); 			% FIR reflection coefficient of blackout screen 													- 						
	addParam(bv, 'tauBlScrNir', 0.01); 			% NIR transmission coefficient of blackout screen 												- 						
	addParam(bv, 'tauBlScrPar', 0.01); 			% PAR transmission coefficient of blackout screen 												- 						
	addParam(bv, 'tauBlScrFir', 0.7); 			% FIR transmission coefficient of blackout screen 												- 						
	addParam(bv, 'cPBlScr', 1.8e3); 			% Specific heat capacity of blackout screen 														J kg^{-1} K^{-1} 		
    addParam(bv, 'hBlScr', 0.35e-3); 			% Thickness of blackout screen 																	m 						
    addParam(bv, 'kBlScr', 0.04e-3); 			% Blackout screen flux coefficient 																m^{3} m^{-2} K^{-2/3} s^{-1} 
	
    
	%% Floor [1]
	% parameter 					            description 																					unit 					nominal value (source)
	addParam(bv, 'epsFlr', 1);			 		% FIR emission coefficient of the floor 														- 						1 [1]
	addParam(bv, 'rhoFlr', 2300); 				% Density of the floor 																			kg m^{-3} 				2300 [1]
	addParam(bv, 'rhoFlrNir', 0.5); 			% NIR reflection coefficient of the floor 														- 						0.5 [1]
	addParam(bv, 'rhoFlrPar', 0.65); 			% PAR reflection coefficient of the floor 														- 						0.65 [1]
	addParam(bv, 'lambdaFlr', 1.7);				% Thermal heat conductivity of the floor 														W m^{-1} K^{-1} 		1.7 [1]
	addParam(bv, 'cPFlr', 0.88e3); 				% Specific heat capacity of the floor 															J kg^{-1} K^{-1} 		0.88e3 [1]
	addParam(bv, 'hFlr', 0.02); 				% Thickness of floor 																			m 						0.02 [1]
	
	%% Soil [1]
	% parameter 					            description 																					unit 					nominal value (source)
    addParam(bv, 'rhoCpSo', 1.73e6); 			% Volumetric heat capacity of the soil 															J m^{-3} K^{-1} 		1.73e6 [1]
	addParam(bv, 'lambdaSo', 0.85); 			% Thermal heat conductivity of the soil layers 													W m^{-1} K^{-1} 		0.85 [1]
	
	%% Heating system [1]
	% parameter 					            description 																					unit 					nominal value (source)
	addParam(bv, 'epsPipe', 0.88); 				% FIR emission coefficient of the heating pipes 												- 						0.88 [1]
	addParam(bv, 'phiPipeE', 51e-3); 			% External diameter of pipes 																	m 						51e-3 [1]
    addParam(bv, 'phiPipeI', 47e-3);            % Internal diameter of pipes 																	m 						47e-3 [1]
    addParam(bv, 'lPipe', 1.875);               % Length of heating pipes per gh floor area 													m m^{-2} 				1.875 [1]
    addParam(bv, 'pBoil', 150*bv.p.aFlr.val);   % Capacity of the heating system                                                                W                       88*p.aFlr (Assumed to be 88 W m^{-2}) [5]
    
	%% Active climate control [1]
	% parameter 					            description 																					unit 					nominal value (source)
    addParam(bv, 'phiExtCo2', 7.2e4);           % Capacity of external CO2 source 																mg s^{-1} 				7.2e4 [1] (for 1.4e4 m^{2}) 2.778 mg s^{-1} m^{-2} [5]
	
	%% Other parameters that depend on previously defined parameters
	    
	% Heat capacity of heating pipes [J K^{-1} m^{-2}]
    % Equation 21 [1]
    bv.p.capPipe = 0.25*pi*bv.p.lPipe*((bv.p.phiPipeE^2-bv.p.phiPipeI^2)*bv.p.rhoSteel*...
        bv.p.cPSteel+bv.p.phiPipeI^2*bv.p.rhoWater*bv.p.cPWater);
    
	% Density of the air [kg m^{-3}]
	% Equation 23 [1]
    bv.p.rhoAir = bv.p.rhoAir0*exp(bv.p.g*bv.p.mAir*bv.p.hElevation/(293.15*bv.p.R));
    
	% Heat capacity of greenhouse objects [J K^{-1} m^{-2}]
	% Equation 22 [1]
    bv.p.capAir = bv.p.hAir*bv.p.rhoAir*bv.p.cPAir;             % air in main compartment
    bv.p.capFlr = bv.p.hFlr*bv.p.rhoFlr*bv.p.cPFlr;             % floor
    bv.p.capSo1 = bv.p.hSo1*bv.p.rhoCpSo;                       % soil layer 1
    bv.p.capSo2 = bv.p.hSo2*bv.p.rhoCpSo;                       % soil layer 2
    bv.p.capSo3 = bv.p.hSo3*bv.p.rhoCpSo;                       % soil layer 3
    bv.p.capSo4 = bv.p.hSo4*bv.p.rhoCpSo;                       % soil layer 4
    bv.p.capSo5 = bv.p.hSo5*bv.p.rhoCpSo;                       % soil layer 5
    bv.p.capThScr = bv.p.hThScr*bv.p.rhoThScr*bv.p.cPThScr;     % thermal screen
    bv.p.capTop = (bv.p.hGh-bv.p.hAir)*bv.p.rhoAir*bv.p.cPAir;  % air in top compartments
    
    bv.p.capBlScr = bv.p.hBlScr*bv.p.rhoBlScr*bv.p.cPBlScr;     % blackout screen

    
	% Capacity for CO2 [m]
	bv.p.capCo2Air = bv.p.hAir;
    bv.p.capCo2Top = bv.p.hGh-bv.p.hAir;
	
	% Surface of pipes for floor area [-]
	% Table 3 [1]
    bv.p.aPipe = pi*bv.p.lPipe*bv.p.phiPipeE;
	
	% View factor from canopy to floor
	% Table 3 [1]
    bv.p.fCanFlr = 1-0.49*pi*bv.p.lPipe*bv.p.phiPipeE;
    
    % Absolute air pressure at given elevation [Pa]
	% See https://www.engineeringtoolbox.com/air-altitude-pressure-d_462.html
    bv.p.pressure = 101325*(1-2.5577e-5*bv.p.hElevation)^5.25588;

	
	%% Canopy photosynthesis [2]
	% parameter 					            description 																					unit 									nominal value (source)
	addParam(bv, 'globJtoUmol', 2.3); 			% Conversion factor from global radiation to PAR (etaGlobPar in [2], but that name has another meaning in [1]) umol{photons} J^{-1} 	2.3 [1]
	addParam(bv, 'j25LeafMax', 210); 			% Maximal rate of electron transport at 25C of the leaf 										umol{e-} m^{-2}{leaf} s^{-1} 			210 [1]
	addParam(bv, 'cGamma', 1.7); 				% Effect of canopy temperature on CO2 compensation point 										umol{co2} mol^{-1}{air} K^{-1} 			1.7 [1]
	addParam(bv, 'etaCo2AirStom', 0.67); 		% Conversion from greenhouse air co2 concentration and stomatal co2 concentration 				umol{co2} mol^{-1}{air} 				0.67 [1]
	addParam(bv, 'eJ', 37e3); 					% Activation energy for Jpot calcualtion 														J mol^{-1} 								37e3 [1]
	addParam(bv, 't25k', 298.15); 				% Reference temperature for Jpot calculation 													K 										298.15 [1]
	addParam(bv, 'S', 710); 					% Enthropy term for Jpot calculation 															J mol^{-1} K^{-1} 						710 [1]
	addParam(bv, 'H', 22e4); 					% Deactivation energy for Jpot calculation 														J mol^{-1} 								22e4 [1]
	addParam(bv, 'theta', 0.7); 				% Degree of curvature of the electron transport rate 											- 										0.7 [1]
	addParam(bv, 'alpha', 0.385); 				% Conversion factor from photons to electrons including efficiency term 						umol{e-} umol^{-1}{photons} 			0.385 [1]
	addParam(bv, 'mCh2o', 30e-3); 				% Molar mass of CH2O 																			mg umol^{-1} 							30e-3
    addParam(bv, 'mCo2', 44e-3);                % Molar mass of CO2                                                                             mg umol^{-1}                            44e-3 
	
    addParam(bv, 'parJtoUmolSun', 4.6); 		% Conversion factor of sun's PAR from J to umol                                                       umol{photons} J^{-1}                    4.6 [1]
    
	addParam(bv, 'laiMax', 3); 					% leaf area index 																				(m^{2}{leaf} m^{-2}{floor}) 			3 [2]
	addParam(bv, 'sla', 2.66e-5); 				% specific leaf area 																			(m^{2}{leaf} mg^{-1}{leaf}			2.66e-5 [2]
	addParam(bv, 'dw', 650e3);                  % assumed crop dry weight 																		{mg{dw} m^{-2}{floor}}                  
    addParam(bv, 'rgr', 3e-6); 					% relative growth rate 																			{kg{dw grown} kg^{-1}{existing dw} s^{-1}} 

    bv.p.cLeafMax = bv.p.laiMax./bv.p.sla;      % maximum leaf size                                                                             mg{leaf} m^{-2}
    
    % cLeafMax is 112e3 mg/m2, the crop max should be about 450e3 mg/m2
    addParam(bv, 'cFruitMax', 300e3);           % maximum fruit size                                                                            mg{fruit} m^{-2}
    
	% Relative size of the organs (based on general assumptions)
    addParam(bv, 'fFruit', 0.5); 				% Fraction of biomass in fruit                                                                  - 										0.5
    addParam(bv, 'fLeaf', 0.4); 				% Fraction of biomass in leaf                                                                   - 										0.4
    addParam(bv, 'fStem', 0.1); 				% Fraction of biomass in stem                                                                   - 										0.1
	    
    addParam(bv, 'cFruitG', 0.27); 				% Fruit growth respiration coefficient 															- 										0.27 [1]
	addParam(bv, 'cLeafG', 0.28); 				% Leaf growth respiration coefficient 															- 										0.28 [1]
	addParam(bv, 'cStemG', 0.3); 				% Stem growth respiration coefficient 															- 										0.3 [1]
	addParam(bv, 'cRgr', 2.85e6); 				% Regression coefficient in maintenance respiration function 									s^{-1} 									2.85e6 [1]
	addParam(bv, 'q10m', 2); 					% Q10 value of temperature effect on maintenance respiration 									- 										2 [1]
	addParam(bv, 'cFruitM', 1.16e-7); 			% Fruit maintenance respiration coefficient 													mg mg^{-1} s^{-1} 						1.16e-7 [1]
	addParam(bv, 'cLeafM', 3.47e-7); 			% Leaf maintenance respiration coefficient 														mg mg^{-1} s^{-1} 						3.47e-7 [1]
	addParam(bv, 'cStemM', 1.47e-7); 			% Stem maintenance respiration coefficient 														mg mg^{-1} s^{-1} 						1.47e-7 [1]
    
    addParam(bv, 'rgFruit', 0.328); 			% Potential fruit growth coefficient                                                            mg m^{-2} s^{-1} 						0.328 [2]
	addParam(bv, 'rgLeaf', 0.095);              % Potential leaf growth coefficient                                                             mg m^{-2} s^{-1} 						0.095 [2]
	addParam(bv, 'rgStem', 0.074);              % Potential stem growth coefficient                                                             mg m^{-2} s^{-1} 						0.074 [2]
    
    
    bv.p.rFruit = bv.p.cFruitG*bv.p.fFruit;     % Growth respiration coefficienct of fruit, assuming constant fraction of fruit                 -
    bv.p.rStem = bv.p.cStemG*bv.p.fStem;        % Growth respiration coefficienct of stem, assuming constant fraction of stem                   -
    bv.p.rLeaf = bv.p.cLeafG*bv.p.fLeaf;        % Growth respiration coefficienct of leaves, assuming constant fraction of leaves               -
    
    %% Carbohydrates buffer
    addParam(bv, 'cBufMax', 20e3);              % Maximum capacity of carbohydrate buffer                                                       mg m^{-2}                               20e3 [2]
    addParam(bv, 'cBufMin', 1e3);               % Minimum capacity of carbohydrate buffer                                                       mg m^{-2}                               1e3 [2]
    addParam(bv, 'tCan24Max', 24.5);            % Inhibition of carbohydrate flow because of high temperatures                                  C                                      24.5 [2]
    addParam(bv, 'tCan24Min', 15);              % Inhibition of carbohydrate flow because of low temperatures                                   C                                      15 [2]
    addParam(bv, 'tCanMax', 34);              % Inhibition of carbohydrate flow because of high instantenous temperatures                       C                                      34 [2]
    addParam(bv, 'tCanMin', 10);              % Inhibition of carbohydrate flow because of low instantenous temperatures                      C                                      10 [2]
    addParam(bv, 'rgOrg', 0.328+0.095+0.074);   % Potential organ growth rate coefficient at 20C                                               mg{CH2O} m^{-2} s^{-1}                  0.328+0.095+0.074 [2]
    addParam(bv, 'rgOrgCorgG', 0.328*0.27+0.095*0.28+0.074*0.3); % Coefficient for growth respiration
    
    %% Crop development
    addParam(bv, 'tEndSum', 1035);              % Temperature sum where crop is fully generative                                                C day                                  1035 [2]
    
    %% Control parameters
        
    %% Rule based control parameters 
    addParam(bv, 'rhMax', 90);                  % upper bound on relative humidity              												[%]                     				90 [4]
	addParam(bv, 'dayThresh', 20); 				% threshold to consider switch from night to day 												[W m^{-2}]								20
    addParam(bv, 'tSpDay', 19.5);               % Heat is on below this point in day           													[C]                                     19.5 [5]
    addParam(bv, 'tSpNight', 16.5);            	% Heat is on below this point in night           												[C]                   					16.5 [5]
    addParam(bv, 'tHeatBand', -1);           	% P-band for heating                              												[C]                   					-1 [6]
    addParam(bv, 'tVentOff', 1); 				% distance from heating setpoint where ventilation stops (even if humidity is too high) 		[C]                   					1 [4]
    addParam(bv, 'tScreenOn', 2); 				% distance from screen setpoint where screen is on (even if humidity is too high)				[C]                   					2 
    addParam(bv, 'thScrSpDay', 5);           	% Screen is closed at day when outdoor is below this temperature 								[C]           							5 [5]
    addParam(bv, 'thScrSpNight', 10);      		% Screen is closed at night when outdoor is below this temperature 								[C]           							10 [5]
    addParam(bv, 'thScrPband', -1);          	% P-band for thermal screen 																	[C] 									-1          
    addParam(bv, 'co2SpDay', 800);            	% Co2 is supplied if co2 is below this point during day 										[ppm]       							800 [4]
    addParam(bv, 'co2Band', -100);              % P-band for co2 supply                                 										[ppm]    								-100   
    addParam(bv, 'heatDeadZone', 5);       		% zone between heating setpoint and ventilation setpoint 										[C]     								5 [4]
    addParam(bv, 'ventHeatPband', 4);     		% P-band for ventilation due to excess heat 													[C] 									4
    addParam(bv, 'ventRhPband', 5);         	% P-band for ventilation due to relative humidity 												[%] 									5
    bv.p.ventThScr = bv.p.rhMax + 2;        	% Relative humidity where thermal screen is forced to open 										[%] 									92
    addParam(bv, 'ventThScrPband', 2);  		% P-band for thermal screen opening due to excess relative humidity 							[%] 									2
    addParam(bv, 'lampsOn', 2);                 % time of day to switch on lamps 																[hours since midnight] 					2
    addParam(bv, 'lampsOff', 18);              	% time of day to switch off lamps 																[hours since midnight] 					18
    addParam(bv, 'lampsOffSun', 200);       	% lamps are switched off if global radiation is above this value 								[W m^{-2}]   							200
    
    %% Grow pipe parameters
    addParam(bv, 'epsGroPipe', 0);              % Emissivity of grow pipes                                                                  [-]                                     0 (no grow pipes)
    
    % There are no grow pipes so these parameters are not important, but
    % they cannot be 0 because the ODE for the grow pipe still exists
    addParam(bv, 'lGroPipe', 1.655); 			% Length of grow pipes per gh floor area                                                    m m^{-2}                                1.25 
    addParam(bv, 'phiGroPipeE', 35e-3); 		% External diameter of grow pipes 															m                                       28e-3 [8]
    addParam(bv, 'phiGroPipeI', (35e-3)-(1.2e-3));% Internal diameter of grow pipes 														m                                       24e-3 
       
    bv.p.aGroPipe = pi*bv.p.lGroPipe*bv.p.phiGroPipeE; % Surface area of pipes for floor area                                                   m^{2}{pipe} m^{-2}{floor}
    addParam(bv, 'pGroBoil', 0);                % Capacity of the grow pipe heating system                                                      W                                       88*p.aFlr (Assumed to be same as pipe rail, but actually based on [5] pBoil+pGroBoil should be 88) 
    
    % Heat capacity of grow pipes [J K^{-1} m^{-2}]
    % Equation 21 [1]
    bv.p.capGroPipe = 0.25*pi*bv.p.lGroPipe*((bv.p.phiGroPipeE^2-bv.p.phiGroPipeI^2)*bv.p.rhoSteel*...
        bv.p.cPSteel+bv.p.phiGroPipeI^2*bv.p.rhoWater*bv.p.cPWater);
    
	%% Lamp parameters - no lamps
    addParam(bv, 'lampsMax', 0);             	% Maximum intensity of lamps																	[W m^{-2}]
	addParam(bv, 'heatCorrection', 0);   		% correction for temperature setpoint when lamps are on 										[C]   									 0
    addParam(bv, 'etaLampPar', 0);  			% fraction of lamp input converted to PAR 														[-]
    addParam(bv, 'etaLampNir', 0);				% fraction of lamp input converted to NIR 														[-]
	addParam(bv, 'tauLampPar', 1);				% transmissivity of lamp layer to PAR															[-]
	addParam(bv, 'rhoLampPar', 0);				% reflectivity of lamp layer to PAR 															[-]
	addParam(bv, 'tauLampNir', 1);				% transmissivity of lamp layer to NIR 															[-]
	addParam(bv, 'rhoLampNir', 0);				% reflectivity of lamp later to NIR 															[-]
	addParam(bv, 'tauLampFir', 1);				% transmissivity of lamp later to FIR 															[-]
	addParam(bv, 'aLamp', 0);					% lamp area 																					[m^{2}{lamp} m^{-2}{floor}]
	addParam(bv, 'epsLampTop', 0); 				% emissivity of top side of lamp 																[-]
	addParam(bv, 'epsLampBottom', 0);			% emissivity of bottom side of lamp 															[-]
	addParam(bv, 'capLamp', 350);				% heat capacity of lamp 																		[J K^{-1} m^{-2}]
    addParam(bv, 'hecLampAir', 0);				% heat exchange coefficient of lamp                                                             [W m^{-2} K^{-1}]
    addParam(bv, 'lampCool', 0);                % fraction of lamp input removed by cooling
    addParam(bv, 'parJtoUmolLamp', 0);          % J to umol conversion of PAR output of lamp                                                    [J{PAR} umol{PAR}^{-1}                  4.93 [Calculated from 10]        
    
    % Interlight parameters - no lamps
    addParam(bv, 'capIntLamp', bv.p.capGroPipe.val);              % heat capacity of lamp 														[J K^{-1} m^{-2}]                       Assumed to be same as grow pipes
    addParam(bv, 'etaIntLampPar', 0);           % fraction of lamp input converted to PAR 														[-]                                     0.65 (calculated based on [7])
    addParam(bv, 'etaIntLampNir', 0);           % fraction of lamp input converted to NIR 														[-]                                     0 [7]   
    addParam(bv, 'aIntLamp', 0);                % interlight lamp area 																			[m^{2}{lamp} m^{-2}{floor}]             0
    addParam(bv, 'epsIntLamp', 0);              % emissivity of interlight [-]                                                                  assumed that lamps act the same as heating pipes
    addParam(bv, 'intLampsMax', 0);           	% Maximum intensity of lamps																	[W m^{-2}]    
	addParam(bv, 'parJtoUmolIntLamp', 0);       % conversion from Joules to umol photons within the PAR output of the interlight
    addParam(bv, 'hecIntLampAir', 0);			% heat exchange coefficient of interlights                                                      [W m^{-2} K^{-1}]

    %% Other parameters
    addParam(bv, 'leakTop', 0.5);               % Fraction of leakage ventilation going from the top                                            [-]                                     0.5 [1]
    addParam(bv, 'minWind', 0.25);              % wind speed where the effect of wind on leakage begins
    addParam(bv, 'simRes', 300);                % resolution of simulation                                                                      [s]                                     300
end

